<?php

use App\Http\Controllers\AuthController;
use App\Http\Controllers\LicenseController;
use App\Http\Controllers\PrescriptionController;
use App\Http\Controllers\PaymentMethodController;
use App\Http\Controllers\SaleTypeController;
use App\Http\Controllers\SubscriptionController;
use App\Http\Controllers\UserController;
use Illuminate\Support\Facades\Route;

// Static asset serving for NativePHP compatibility
Route::get('/plugins/{path}', function ($path) {
    $filePath = public_path("plugins/{$path}");
    
    if (!file_exists($filePath)) {
        abort(404);
    }
    
    $mimeType = match (pathinfo($filePath, PATHINFO_EXTENSION)) {
        'js' => 'application/javascript',
        'css' => 'text/css',
        'png' => 'image/png',
        'jpg', 'jpeg' => 'image/jpeg',
        'gif' => 'image/gif',
        'svg' => 'image/svg+xml',
        'woff' => 'font/woff',
        'woff2' => 'font/woff2',
        'ttf' => 'font/ttf',
        default => 'application/octet-stream',
    };
    
    return response()->file($filePath, [
        'Content-Type' => $mimeType,
        'Cache-Control' => 'public, max-age=31536000', // 1 year cache
    ]);
})->where('path', '.*');

Route::middleware(['auth', 'app.licensed'])->get('/', App\Livewire\Home::class)->name('home');
Route::middleware('auth')->post('/logout', [AuthController::class, 'logout'])->name('logout');
Route::middleware(['auth', 'prevent.license.access'])->get('/license', App\Livewire\License::class)->name('license');
Route::middleware(['auth', 'prevent.license.access'])->post('/license', [LicenseController::class, 'store'])->name('license.store');

// Legal pages - accessible to all authenticated users
Route::middleware('auth')->group(function () {
    Route::get('/terms', function () {
        return view('legal.terms');
    })->name('terms');
    
    Route::get('/privacy', function () {
        return view('legal.privacy');
    })->name('privacy');
});

Route::middleware('guest')->group(function () {
    Route::get('/login', App\Livewire\Auth\Login::class)->name('login');
    Route::get('/register', App\Livewire\Auth\Register::class)->name('register');

    // Password reset routes using Livewire components
    Route::get('/password/reset', App\Livewire\Auth\PasswordReset::class)->name('password.request');
    Route::get('/password/reset/{token}', \App\Livewire\Auth\PasswordResetForm::class)->name('password.reset');

    // Email set-password link (signed)
    Route::get('/users/set-password/{user}', App\Livewire\Auth\SetPassword::class)
        ->middleware('signed')
        ->name('users.set-password');
});

// OTP verification - accessible to authenticated users who need to verify
Route::get('/otp', App\Livewire\Auth\OtpVerify::class)->name('otp.verify');

Route::middleware(['auth', 'app.licensed'])->group(function () {

    // CUSTOMERS
    Route::name('customers.')->prefix('customers')->group(function () {
        Route::get('/', App\Livewire\Customers\Index::class)->name('index');
        Route::post('/', [\App\Http\Controllers\CustomerController::class, 'store'])->name('store');
        Route::get('/{customer}', App\Livewire\Customers\Show::class)->name('show');
        Route::get('/{customer}/edit', App\Livewire\Customers\Edit::class)->name('edit');
        Route::put('/{customer}', [\App\Http\Controllers\CustomerController::class, 'update'])->name('update');
        Route::post('/{customer}/activate', [\App\Http\Controllers\CustomerController::class, 'activate'])->name('activate');
        Route::post('/{customer}/deactivate', [\App\Http\Controllers\CustomerController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{customer}', [\App\Http\Controllers\CustomerController::class, 'destroy'])->name('destroy');

        // Customer Sales
        Route::name('sales.')->prefix('{customer}/sales')->group(function () {
            Route::get('/', App\Livewire\Customers\Sales::class)->name('index');
        });

        // Customer Debts
        Route::name('debts.')->prefix('{customer}/debts')->group(function () {
            Route::get('/', App\Livewire\Customers\Debts::class)->name('index');
        });
    });

    // SALES
    Route::name('sales.')->prefix('sales')->group(function () {
        Route::get('/', App\Livewire\Sales\Index::class)->name('index');
        Route::get('/create', App\Livewire\Sales\Create::class)->name('create');
        Route::get('/{sale}', App\Livewire\Sales\Show::class)->name('show');
        Route::get('/{sale}/edit', App\Livewire\Sales\Edit::class)->name('edit');
        Route::put('/{sale}', [\App\Http\Controllers\SaleController::class, 'update'])->name('update');
        Route::delete('/{sale}', [\App\Http\Controllers\SaleController::class, 'destroy'])->name('destroy');
        Route::put('/{sale}/cancel', [\App\Http\Controllers\SaleController::class, 'cancel'])->name('cancel');
        Route::get('/{sale}/print', [\App\Http\Controllers\SaleController::class, 'print'])->name('print');
    });

    // PRESCRIPTIONS
    Route::name('prescriptions.')->prefix('prescriptions')->group(function () {
        Route::get('/', App\Livewire\Prescriptions\Index::class)->name('index');
        Route::post('/{reference}/approve', [PrescriptionController::class, 'approve'])->name('approve');
        Route::post('/{reference}/cancel', [PrescriptionController::class, 'cancel'])->name('cancel');
        Route::post('/{reference}/finish', [PrescriptionController::class, 'finish'])->name('finish');
        Route::get('/{reference}', App\Livewire\Prescriptions\Show::class)->name('show');
    });

    // CONVERSATIONS
    Route::name('conversations.')->prefix('conversations')->group(function () {
        Route::get('/', App\Livewire\Conversations\Index::class)->name('index');
        Route::get('/{slug}', App\Livewire\Conversations\Show::class)->name('show');
    });

    // DEBTS
    Route::name('debts.')->prefix('debts')->group(function () {
        Route::get('/', App\Livewire\Debts\Index::class)->name('index');
        Route::get('/{debt}', App\Livewire\Debts\Show::class)->name('show');
        Route::get('/{debt}/edit', App\Livewire\Debts\Edit::class)->name('edit');
        Route::put('/{debt}', [\App\Http\Controllers\DebtController::class, 'update'])->name('update');
        Route::delete('/{debt}', [\App\Http\Controllers\DebtController::class, 'destroy'])->name('destroy');
        Route::post('/{debt}/payments', [\App\Http\Controllers\DebtController::class, 'addPayment'])->name('add-payment');
    });

    // EXPENSES
    Route::name('expenses.')->prefix('expenses')->group(function () {
        Route::get('/', App\Livewire\Expenses\Index::class)->name('index');
        Route::post('/', [\App\Http\Controllers\ExpenseController::class, 'store'])->name('store');
        Route::get('/{expense}', App\Livewire\Expenses\Show::class)->name('show');
        Route::get('/{expense}/edit', App\Livewire\Expenses\Edit::class)->name('edit');
        Route::put('/{expense}/approve', [\App\Http\Controllers\ExpenseController::class, 'approve'])->name('approve');
        Route::put('/{expense}/reject', [\App\Http\Controllers\ExpenseController::class, 'reject'])->name('reject');
        Route::put('/{expense}/mark-paid', [\App\Http\Controllers\ExpenseController::class, 'markPaid'])->name('mark-paid');
        Route::delete('/{expense}', [\App\Http\Controllers\ExpenseController::class, 'destroy'])->name('destroy');
    });

    // EXPENSE CATEGORIES
    Route::name('expense-categories.')->prefix('expense-categories')->group(function () {
        Route::get('/', App\Livewire\ExpenseCategories\Index::class)->name('index');
        Route::get('/{category}', App\Livewire\ExpenseCategories\Show::class)->name('show');
        Route::get('/{category}/edit', App\Livewire\ExpenseCategories\Edit::class)->name('edit');
        Route::post('/', [\App\Http\Controllers\ExpenseCategoryController::class, 'store'])->name('store');
        Route::put('/{category}/activate', [\App\Http\Controllers\ExpenseCategoryController::class, 'activate'])->name('activate');
        Route::put('/{category}/deactivate', [\App\Http\Controllers\ExpenseCategoryController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{category}', [\App\Http\Controllers\ExpenseCategoryController::class, 'destroy'])->name('destroy');
    });

    // SUPPLIERS
    Route::name('suppliers.')->prefix('suppliers')->group(function () {
        Route::get('/', App\Livewire\Suppliers\Index::class)->name('index');
        Route::get('/{supplier}', App\Livewire\Suppliers\Show::class)->name('show');
        Route::get('/{supplier}/edit', App\Livewire\Suppliers\Edit::class)->name('edit');
        Route::post('/', [\App\Http\Controllers\SupplierController::class, 'store'])->name('store');
        Route::put('/{supplier}', [\App\Http\Controllers\SupplierController::class, 'update'])->name('update');
        Route::post('/{supplier}/activate', [\App\Http\Controllers\SupplierController::class, 'activate'])->name('activate');
        Route::post('/{supplier}/deactivate', [\App\Http\Controllers\SupplierController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{supplier}', [\App\Http\Controllers\SupplierController::class, 'destroy'])->name('destroy');
    });

    // UNITS
    Route::name('units.')->prefix('units')->group(function () {
        Route::get('/', App\Livewire\Units\Index::class)->name('index');
        Route::get('/{unit}', App\Livewire\Units\Show::class)->name('show');
        Route::get('/{unit}/edit', App\Livewire\Units\Edit::class)->name('edit');
        Route::post('/', [\App\Http\Controllers\UnitController::class, 'store'])->name('store');
        Route::put('/{unit}', [\App\Http\Controllers\UnitController::class, 'update'])->name('update');
        Route::delete('/{unit}', [\App\Http\Controllers\UnitController::class, 'destroy'])->name('destroy');
    });

    // CATEGORIES
    Route::name('categories.')->prefix('categories')->group(function () {
        Route::get('/', App\Livewire\Categories\Index::class)->name('index');
        Route::get('/{category}', App\Livewire\Categories\Show::class)->name('show');
        Route::get('/{category}/edit', App\Livewire\Categories\Edit::class)->name('edit');
        Route::post('/', [\App\Http\Controllers\CategoryController::class, 'store'])->name('store');
        Route::put('/{category}', [\App\Http\Controllers\CategoryController::class, 'update'])->name('update');
        Route::delete('/{category}', [\App\Http\Controllers\CategoryController::class, 'destroy'])->name('destroy');
    });

    // STOCK ADJUSTMENTS
    Route::name('stock-adjustments.')->prefix('stock-adjustments')->group(function () {
        Route::get('/', App\Livewire\StockAdjustments\Index::class)->name('index');
        Route::get('/{stockAdjustment}', App\Livewire\StockAdjustments\Show::class)->name('show');
        Route::get('/{stockAdjustment}/edit', App\Livewire\StockAdjustments\Edit::class)->name('edit');
        Route::post('/', [\App\Http\Controllers\StockAdjustmentController::class, 'store'])->name('store');
        Route::put('/{stockAdjustment}', [\App\Http\Controllers\StockAdjustmentController::class, 'update'])->name('update');
        Route::post('/{stockAdjustment}/apply', [\App\Http\Controllers\StockAdjustmentController::class, 'apply'])->name('apply');
        Route::get('/{stockAdjustment}/print', [\App\Http\Controllers\StockAdjustmentController::class, 'print'])->name('print');
        Route::delete('/{stockAdjustment}', [\App\Http\Controllers\StockAdjustmentController::class, 'destroy'])->name('destroy');
    });

    // STOCK ENTRIES
    Route::name('stock-entries.')->prefix('stock-entries')->group(function () {
        Route::get('/', App\Livewire\StockEntries\Index::class)->name('index');
        Route::get('/create', [\App\Http\Controllers\StockEntryController::class, 'create'])->name('create');
        Route::get('/{stockEntry}', App\Livewire\StockEntries\Show::class)->name('show');
        Route::get('/{stockEntry}/edit', App\Livewire\StockEntries\Edit::class)->name('edit');
        Route::post('/', [\App\Http\Controllers\StockEntryController::class, 'store'])->name('store');
        Route::put('/{stockEntry}', [\App\Http\Controllers\StockEntryController::class, 'update'])->name('update');
        Route::post('/{stockEntry}/complete', [\App\Http\Controllers\StockEntryController::class, 'complete'])->name('complete');
        Route::post('/{stockEntry}/cancel', [\App\Http\Controllers\StockEntryController::class, 'cancel'])->name('cancel');
        Route::delete('/{stockEntry}', [\App\Http\Controllers\StockEntryController::class, 'destroy'])->name('destroy');
    });

    // PURCHASE ORDERS
    Route::name('purchase-orders.')->prefix('purchase-orders')->group(function () {
        Route::get('/', App\Livewire\PurchaseOrders\Index::class)->name('index');
        Route::get('/{purchaseOrder}', App\Livewire\PurchaseOrders\Show::class)->name('show');
        Route::get('/{purchaseOrder}/edit', App\Livewire\PurchaseOrders\Edit::class)->name('edit');
        Route::get('/{purchaseOrder}/receive', App\Livewire\PurchaseOrders\Receive::class)->name('receive-form');
        Route::get('/{purchaseOrder}/print', [\App\Http\Controllers\PurchaseOrderController::class, 'print'])->name('print');
        Route::post('/', [\App\Http\Controllers\PurchaseOrderController::class, 'store'])->name('store');
        Route::put('/{purchaseOrder}', [\App\Http\Controllers\PurchaseOrderController::class, 'update'])->name('update');
        Route::post('/{purchaseOrder}/activate', [\App\Http\Controllers\PurchaseOrderController::class, 'activate'])->name('activate');
        Route::post('/{purchaseOrder}/reactivate', [\App\Http\Controllers\PurchaseOrderController::class, 'reactivate'])->name('reactivate');
        Route::post('/{purchaseOrder}/cancel', [\App\Http\Controllers\PurchaseOrderController::class, 'cancel'])->name('cancel');
        Route::post('/{purchaseOrder}/receive', [\App\Http\Controllers\PurchaseOrderController::class, 'receive'])->name('receive');
        Route::delete('/{purchaseOrder}', [\App\Http\Controllers\PurchaseOrderController::class, 'destroy'])->name('destroy');
    });

    // PRODUCTS
    Route::name('products.')->prefix('products')->group(function () {
        Route::get('/', App\Livewire\Products\Index::class)->name('index');
        Route::get('/{product}', App\Livewire\Products\Show::class)->name('show');
        Route::get('/{product}/edit', App\Livewire\Products\Edit::class)->name('edit');
        Route::post('/', [\App\Http\Controllers\ProductController::class, 'store'])->name('store');
        Route::put('/{product}', [\App\Http\Controllers\ProductController::class, 'update'])->name('update');
        Route::post('/{product}/activate', [\App\Http\Controllers\ProductController::class, 'activate'])->name('activate');
        Route::post('/{product}/deactivate', [\App\Http\Controllers\ProductController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{product}', [\App\Http\Controllers\ProductController::class, 'destroy'])->name('destroy');
    });

    // MERCHANT
    Route::name('merchant.')->prefix('merchant')->group(function () {
        Route::get('/', App\Livewire\Merchant\Show::class)->name('show');
        Route::get('/edit', App\Livewire\Merchant\Edit::class)->name('edit');
    });

    // USERS
    Route::name('users.')->prefix('users')->group(function () {
        Route::post('/', [UserController::class, 'store'])->name('store');
        Route::get('/', App\Livewire\Users\Index::class)->name('index');
        Route::get('/{user}/edit', App\Livewire\Users\Edit::class)->name('edit');
        Route::get('/{user}', App\Livewire\Users\Show::class)->name('show');
        Route::get('/{user}/activities', App\Livewire\Users\Activities::class)->name('activities');
        Route::post('/{user}/activate', [UserController::class, 'activate'])->name('activate');
        Route::post('/{user}/deactivate', [UserController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{user}', [UserController::class, 'destroy'])->name('destroy');
    });

    // SALE TYPES
    Route::name('sale-types.')->prefix('sale-types')->group(function () {
        Route::get('/', App\Livewire\SaleTypes\Index::class)->name('index');
        Route::get('/{saleType}', App\Livewire\SaleTypes\Show::class)->name('show');
        Route::get('/{saleType}/edit', App\Livewire\SaleTypes\Edit::class)->name('edit');
        Route::post('/', [SaleTypeController::class, 'store'])->name('store');
        Route::put('/{saleType}', [SaleTypeController::class, 'update'])->name('update');
        Route::post('/{saleType}/activate', [SaleTypeController::class, 'activate'])->name('activate');
        Route::post('/{saleType}/deactivate', [SaleTypeController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{saleType}', [SaleTypeController::class, 'destroy'])->name('destroy');
    });

    // PAYMENT METHODS
    Route::name('payment-methods.')->prefix('payment-methods')->group(function () {
        Route::get('/', App\Livewire\PaymentMethods\Index::class)->name('index');
        Route::get('/{paymentMethod}', App\Livewire\PaymentMethods\Show::class)->name('show');
        Route::get('/{paymentMethod}/edit', App\Livewire\PaymentMethods\Edit::class)->name('edit');
        Route::post('/', [PaymentMethodController::class, 'store'])->name('store');
        Route::put('/{paymentMethod}', [PaymentMethodController::class, 'update'])->name('update');
        Route::post('/{paymentMethod}/activate', [PaymentMethodController::class, 'activate'])->name('activate');
        Route::post('/{paymentMethod}/deactivate', [PaymentMethodController::class, 'deactivate'])->name('deactivate');
        Route::delete('/{paymentMethod}', [PaymentMethodController::class, 'destroy'])->name('destroy');
    });

    // REPORTS
    Route::name('reports.')->prefix('reports')->group(function () {
        Route::get('/stock-report', App\Livewire\Reports\StockReport::class)->name('stock-report');
        Route::get('/stock-report/guide', App\Livewire\Reports\StockReportGuide::class)->name('stock-report.guide');
        Route::post('/stock-report/export', [App\Http\Controllers\StockReportController::class, 'export'])->name('stock-report.export');
        Route::post('/stock-report/print', [App\Http\Controllers\StockReportController::class, 'print'])->name('stock-report.print');

        Route::get('/sales-performance', App\Livewire\Reports\SalesPerformance::class)->name('sales-performance');
        Route::get('/sales-performance/guide', App\Livewire\Reports\SalesPerformanceGuide::class)->name('sales-performance.guide');
        Route::post('/sales-performance/export', [App\Http\Controllers\SalesDashboardController::class, 'export'])->name('sales-performance.export');
        Route::post('/sales-performance/print', [App\Http\Controllers\SalesDashboardController::class, 'print'])->name('sales-performance.print');

        Route::get('/inventory-intelligence', App\Livewire\Reports\InventoryIntelligence::class)->name('inventory-intelligence');
        Route::get('/inventory-intelligence/guide', App\Livewire\Reports\InventoryIntelligenceGuide::class)->name('inventory-intelligence.guide');
        Route::post('/inventory-intelligence/export', [App\Http\Controllers\InventoryDashboardController::class, 'export'])->name('inventory-intelligence.export');
        Route::post('/inventory-intelligence/print', [App\Http\Controllers\InventoryDashboardController::class, 'print'])->name('inventory-intelligence.print');

        Route::get('/business-intelligence', App\Livewire\Reports\BusinessIntelligence::class)->name('business-intelligence');
        Route::get('/business-intelligence/guide', App\Livewire\Reports\BusinessIntelligenceGuide::class)->name('business-intelligence.guide');
        Route::post('/business-intelligence/export', [App\Http\Controllers\BusinessDashboardController::class, 'export'])->name('business-intelligence.export');
        Route::post('/business-intelligence/print', [App\Http\Controllers\BusinessDashboardController::class, 'print'])->name('business-intelligence.print');
    });

    // ACCOUNT
    Route::name('account.')->prefix('account')->group(function () {
        Route::get('/', App\Livewire\Account\Index::class)->name('index');
        Route::get('/edit', App\Livewire\Account\Edit::class)->name('edit');
        Route::get('/password', App\Livewire\Account\Password::class)->name('password');
    });

    // SUBSCRIPTIONS
    Route::name('subscriptions.')->prefix('subscriptions')->group(function () {
        Route::get('/', App\Livewire\Subscriptions\Index::class)->name('index');
        Route::get('/{subscription}', App\Livewire\Subscriptions\Show::class)->name('show');
        Route::post('/', [SubscriptionController::class, 'store'])->name('store');
    });
});
