<?php

namespace Database\Factories;

use App\Models\Subscription;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\Subscription>
 */
class SubscriptionFactory extends Factory
{
    /**
     * The name of the factory's corresponding model.
     *
     * @var string
     */
    protected $model = Subscription::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $startDate = $this->faker->dateTimeBetween('-1 year', 'now');
        $durationInDays = $this->faker->numberBetween(30, 365);
        $endDate = (clone $startDate)->modify("+{$durationInDays} days");
        
        return [
            'slug' => Str::orderedUuid(),
            'category' => $this->faker->randomElement(['Premium', 'Basic', 'Enterprise']),
            'code' => 'SUB' . $this->faker->unique()->numerify('########'),
            'start_date' => $startDate,
            'end_date' => $endDate,
            'duration_in_days' => $durationInDays,
            'status' => $this->faker->randomElement([
                Subscription::STATUS_ACTIVE,
                Subscription::STATUS_INACTIVE,
                Subscription::STATUS_EXPIRED,
            ]),
            'activated_at' => $this->faker->optional(0.7)->dateTimeBetween('-1 year', 'now'),
        ];
    }

    /**
     * Indicate that the subscription is active.
     */
    public function active(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => Subscription::STATUS_ACTIVE,
            'activated_at' => now(),
        ]);
    }

    /**
     * Indicate that the subscription is inactive.
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => Subscription::STATUS_INACTIVE,
            'activated_at' => null,
        ]);
    }

    /**
     * Indicate that the subscription is expired.
     */
    public function expired(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => Subscription::STATUS_EXPIRED,
            'start_date' => $this->faker->dateTimeBetween('-2 years', '-1 year'),
            'duration_in_days' => 365,
            'activated_at' => $this->faker->dateTimeBetween('-2 years', '-1 year'),
        ]);
    }
}
