<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class StockEntry extends Model
{
    const SOURCE_TYPE_PURCHASE = 'purchase';

    const SOURCE_TYPE_ADJUSTMENT = 'adjustment';

    const SOURCE_TYPE_TRANSFER = 'transfer';

    const SOURCE_TYPE_RETURN = 'return';

    const STATUS_DRAFT = 'draft';

    const STATUS_COMPLETED = 'completed';

    const STATUS_CANCELLED = 'cancelled';

    protected $guarded = [];

    protected $casts = [
        'quantity' => 'integer',
        'remaining_quantity' => 'integer',
        'purchase_price' => 'decimal:2',
        'expiry_date' => 'date',
    ];

    public function getRouteKeyName()
    {
        return 'reference';
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function purchaseOrder(): BelongsTo
    {
        return $this->belongsTo(PurchaseOrder::class);
    }

    public function purchaseOrderItem(): BelongsTo
    {
        return $this->belongsTo(PurchaseOrderItem::class);
    }

    public function getSourceTypeLabel(): string
    {
        return match ($this->source_type) {
            self::SOURCE_TYPE_PURCHASE => 'Achat',
            self::SOURCE_TYPE_ADJUSTMENT => 'Ajustement',
            self::SOURCE_TYPE_TRANSFER => 'Transfert',
            self::SOURCE_TYPE_RETURN => 'Retour',
            default => 'Inconnu',
        };
    }

    public function getSourceTypeBadge(): string
    {
        if ($this->source_type == self::SOURCE_TYPE_ADJUSTMENT) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-blue-100 text-blue-800 border border-blue-500 min-w-30 justify-center">
                <i class="ph ph-arrows-out-line mr-1"></i>
                Ajustement
            </span>';
        } elseif ($this->source_type == self::SOURCE_TYPE_TRANSFER) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-gray-100 text-gray-800 border border-gray-500 min-w-30 justify-center">
                <i class="ph ph-arrow-right mr-1"></i>
                Transfert
            </span>';
        } elseif ($this->source_type == self::SOURCE_TYPE_RETURN) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-red-100 text-red-800 border border-red-500 min-w-30 justify-center">
                <i class="ph ph-check-circle mr-1"></i>
                Retour
            </span>';
        } else {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-green-100 text-green-800 border border-green-500 min-w-30 justify-center">
                <i class="ph ph-check-circle mr-1"></i>
                Achat
            </span>';
        }

    }

    public function getTotalAmount(): float
    {
        return $this->quantity * ($this->purchase_price ?? 0);
    }

    public function isExpired(): bool
    {
        if (! $this->expiry_date) {
            return false;
        }

        return $this->expiry_date->isPast();
    }

    public function getExpiryStatusBadge(): string
    {
        if (! $this->expiry_date) {
            return '<span class="inline-block px-3 py-1 border border-gray-500 bg-gray-100 text-xs font-medium rounded text-gray-700 min-w-30 justify-center">Aucune date</span>';
        }

        if ($this->isExpired()) {
            return '<span class="inline-block px-3 py-1 border border-red-500 bg-red-100 text-xs font-medium rounded text-red-700 min-w-30 justify-center">Expiré</span>';
        }

        $daysUntilExpiry = now()->diffInDays($this->expiry_date, false);

        if ($daysUntilExpiry <= 30) {
            return '<span class="inline-block px-3 py-1 border border-orange-500 bg-orange-100 text-xs font-medium rounded text-orange-700 min-w-30 justify-center">Expire bientôt</span>';
        }

        return '<span class="inline-block px-3 py-1 border border-green-500 bg-green-100 text-xs font-medium rounded text-green-700 min-w-30 justify-center">Valide</span>';
    }

    public static function generateUniqueReference()
    {
        $reference = 'SE-'.date('Ymd').'-'.str_pad(self::count() + 1, 3, '0', STR_PAD_LEFT);

        return $reference;
    }

    public static function generateUniqueBatchNumber()
    {
        $batchNumber = 'BATCH_NO-'.date('Ymd').'-'.str_pad(self::count() + 1, 3, '0', STR_PAD_LEFT);

        return $batchNumber;
    }

    public function getStatusLabel()
    {
        return match ($this->status) {
            self::STATUS_DRAFT => 'Brouillon',
            self::STATUS_COMPLETED => 'Complété',
            self::STATUS_CANCELLED => 'Annulé',
            default => 'Inconnu',
        };
    }

    public function getStatusBadge()
    {
        if ($this->status == self::STATUS_DRAFT) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-blue-100 text-blue-800 border border-blue-500 min-w-30 justify-center">
            <i class="ph ph-arrows-out-line mr-1"></i>
            Brouillon</span>';
        } elseif ($this->status == self::STATUS_COMPLETED) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-green-100 text-green-800 border border-green-500 min-w-30 justify-center">
            <i class="ph ph-check-circle mr-1"></i>
            Complété</span>';
        } elseif ($this->status == self::STATUS_CANCELLED) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-red-100 text-red-800 border border-red-500 min-w-30 justify-center">
            <i class="ph ph-x-circle mr-1"></i>
            Annulé</span>';
        }
        return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-gray-100 text-gray-800 border border-gray-500 min-w-30 justify-center">
        <i class="ph ph-x-circle mr-1"></i>
        Inconnu</span>';
    }
}
