<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class StockAdjustment extends Model
{
    use HasFactory;

    protected $guarded = [];

    protected function casts(): array
    {
        return [
            'applied_at' => 'datetime',
        ];
    }

    public const TYPE_DECREASE = 'decrease';

    public const TYPE_INCREASE = 'increase';

    public const STATUS_DRAFT = 0;

    public const STATUS_SUBMITTED = 1;

    public const STATUS_APPLIED = 2;

    public const STATUS_CANCELLED = 3;

    const TYPES = [
        ['id' => 'decrease', 'name' => 'Diminution'],
        ['id' => 'increase', 'name' => 'Augmentation'],
    ];

    const STATUSES = [
        ['id' => 'draft', 'name' => 'Brouillon'],
        ['id' => 'submitted', 'name' => 'Soumis'],
        ['id' => 'applied', 'name' => 'Appliqué'],
        ['id' => 'cancelled', 'name' => 'Annulé'],
    ];

    public function getRouteKeyName()
    {
        return 'reference';
    }

    public static function generateReference(): string
    {
        $prefix = 'ADJ-'.date('Ymd').'-';
        $count = static::whereDate('created_at', now()->toDateString())->count() + 1;

        return $prefix.str_pad($count, 3, '0', STR_PAD_LEFT);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function appliedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'applied_by');
    }

    public function items(): HasMany
    {
        return $this->hasMany(StockAdjustmentItem::class);
    }

    public function getStatusBadge()
    {
        if ($this->status == self::STATUS_APPLIED) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-green-100 text-green-800 border border-green-500 min-w-30 justify-center">
                <i class="ph ph-check-circle mr-1"></i>
                Appliqué
            </span>';
        } elseif ($this->status == self::STATUS_DRAFT) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-gray-100 text-gray-800 border border-gray-500 min-w-30 justify-center">
                <i class="ph ph-clock mr-1"></i>
                Brouillon
            </span>';
        } elseif ($this->status == self::STATUS_SUBMITTED) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-yellow-100 text-yellow-800 border border-yellow-500 min-w-30 justify-center">
                <i class="ph ph-pencil mr-1"></i>
                Soumis
            </span>';
        } elseif ($this->status == self::STATUS_CANCELLED) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-red-100 text-red-800 border border-red-500 min-w-30 justify-center">
                <i class="ph ph-x-circle mr-1"></i>
                Annulé
            </span>';
        }
    }

    public function getStatusName()
    {
        if ($this->status == self::STATUS_APPLIED) {
            return 'Appliqué';
        } elseif ($this->status == self::STATUS_DRAFT) {
            return 'Brouillon';
        }
    }

    public function getTypeName()
    {
        if ($this->type == self::TYPE_DECREASE) {
            return 'Diminution';
        } elseif ($this->type == self::TYPE_INCREASE) {
            return 'Augmentation';
        }
    }
}
