<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Setting extends Model
{
    protected $guarded = [];

    const DEFAULT_CURRENCY = 'CDF';

    const DEFAULT_CURRENCY_SYMBOL = 'FC';

    const DEFAULT_TIMEZONE = 'Africa/Kinshasa';

    const DEFAULT_PAYMENT_METHOD_ID = 1;

    const DEFAULT_PAYMENT_METHOD_CASH_ID = 1;

    const DEFAULT_PAYMENT_METHOD_DEBT_ID = 2;

    const DEFAULT_SALE_TYPE_ID = 1;

    protected $casts = [
        'has_active_subscription' => 'boolean',
    ];

    /**
     * Boot method to automatically set merchant_slug
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($setting) {
            if (empty($setting->merchant_slug) && ! empty($setting->merchant_name)) {
                $setting->merchant_slug = static::generateSlug($setting->merchant_name);
            }
        });

        static::updating(function ($setting) {
            if ($setting->isDirty('merchant_name') && empty($setting->merchant_slug)) {
                $setting->merchant_slug = static::generateSlug($setting->merchant_name);
            }
        });
    }

    /**
     * Generate a unique slug for merchant name
     */
    public static function generateSlug($name, $suffix = '')
    {
        $baseSlug = Str::slug($name);
        $slug = $baseSlug.$suffix;

        $counter = 1;
        while (static::where('merchant_slug', $slug)->exists()) {
            $slug = $baseSlug.'-'.$counter;
            $counter++;
        }

        return $slug;
    }

    /**
     * Get the province that owns the setting
     */
    public function province()
    {
        return $this->belongsTo(Province::class, 'merchant_province_id');
    }

    /**
     * Get the city that owns the setting
     */
    public function city()
    {
        return $this->belongsTo(City::class, 'merchant_city_id');
    }

    /**
     * Get a setting value by key
     */
    public static function getValue($key, $default = null)
    {
        $setting = static::where('key', $key)->first();

        return $setting ? $setting->value : $default;
    }

    /**
     * Set a setting value
     */
    public static function setValue($key, $value)
    {
        static::updateOrCreate(['key' => $key], ['value' => $value]);
    }

    /**
     * Get merchant name
     */
    public static function merchantName()
    {
        $setting = static::first();

        return $setting ? $setting->merchant_name : 'My Pharmacy';
    }

    /**
     * Get merchant address
     */
    public static function merchantAddress()
    {
        $setting = static::first();

        return $setting ? $setting->merchant_address : null;
    }

    /**
     * Get merchant phone
     */
    public static function merchantPhone()
    {
        $setting = static::first();

        return $setting ? $setting->merchant_phone : null;
    }

    /**
     * Get merchant email
     */
    public static function merchantEmail()
    {
        $setting = static::first();

        return $setting ? $setting->merchant_email : null;
    }

    /**
     * Get merchant logo
     */
    public static function merchantLogo()
    {
        $setting = static::first();

        return $setting ? $setting->merchant_logo : null;
    }

    /**
     * Get merchant primary color
     */
    public static function merchantColorPrimary()
    {
        return static::getValue('merchant_color_primary', '#3B82F6');
    }

    /**
     * Get merchant secondary color
     */
    public static function merchantColorSecondary()
    {
        return static::getValue('merchant_color_secondary', '#6B7280');
    }

    /**
     * Get merchant type
     */
    public static function merchantType()
    {
        $setting = static::first();

        return $setting ? $setting->merchant_type : 'pharmacie';
    }

    /**
     * Get merchant tax number
     */
    public static function merchantTaxNumber()
    {
        $setting = static::first();

        return $setting ? $setting->merchant_tax_number : null;
    }

    /**
     * Get merchant currency
     */
    public static function merchantCurrency()
    {
        return static::getValue('merchant_currency', self::DEFAULT_CURRENCY);
    }

    /**
     * Get merchant currency symbol
     */
    public static function merchantCurrencySymbol()
    {
        return static::getValue('merchant_currency_symbol', self::DEFAULT_CURRENCY_SYMBOL);
    }

    /**
     * Get merchant timezone
     */
    public static function merchantTimezone()
    {
        return static::getValue('merchant_timezone', self::DEFAULT_TIMEZONE);
    }

    /**
     * Check if has active subscription
     */
    public static function hasActiveSubscription()
    {
        $setting = static::first();

        return $setting ? $setting->has_active_subscription : false;
    }
}
