<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class ExpenseCategory extends Model
{
    use HasFactory;

    protected $guarded = [];

    const STATUS_INACTIVE = 0;

    const STATUS_ACTIVE = 1;

    const STATUS_SUSPENDED = 2;

    protected $casts = [
        'status' => 'integer',
    ];

    public function getRouteKeyName()
    {
        return 'slug'; // Using slug for route binding as requested
    }

    // Relationships
    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    public function scopeSuspended($query)
    {
        return $query->where('status', self::STATUS_SUSPENDED);
    }

    public function scopeInactive($query)
    {
        return $query->where('status', self::STATUS_INACTIVE);
    }

    // Methods
    public function getStatusBadge()
    {
        if ($this->status == self::STATUS_ACTIVE) {
            return '<span class="inline-block px-3 py-1 border border-green-500 bg-green-100 text-xs font-medium rounded text-green-700 text-center min-w-30"> <i class="ph ph-check-circle mr-1"></i>Actif</span>';
        } elseif ($this->status == self::STATUS_INACTIVE) {
            return '<span class="inline-block px-3 py-1 border border-red-500 bg-red-100 text-xs font-medium rounded text-red-700 text-center min-w-30"> <i class="ph ph-x-circle mr-1"></i>Inactif</span>';
        } elseif ($this->status == self::STATUS_SUSPENDED) {
            return '<span class="inline-block px-3 py-1 border border-yellow-500 bg-yellow-100 text-xs font-medium rounded text-yellow-700 text-center min-w-30"> <i class="ph ph-x-circle mr-1"></i>Suspendu</span>';
        } else {
            return '<span class="inline-block px-3 py-1 border border-gray-500 bg-gray-100 text-xs font-medium rounded text-gray-700 text-center min-w-30"> <i class="ph ph-x-circle mr-1"></i>Inconnu</span>';
        }
    }

    public function getExpensesCountThisMonthAttribute()
    {
        return $this->expenses()->where('created_at', '>=', now()->startOfMonth())->count();
    }

    public function getTotalExpensesAttribute()
    {
        return $this->expenses()->sum('amount') ?? 0;
    }

    public function getTotalExpensesThisMonthAttribute()
    {
        return $this->expenses()->where('created_at', '>=', now()->startOfMonth())->sum('amount') ?? 0;
    }

    public function getExpensesCountAttribute()
    {
        return $this->expenses()->count() ?? 0;
    }

    public function isActive()
    {
        return $this->status == self::STATUS_ACTIVE;
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($category) {
            if (empty($category->slug)) {
                $category->slug = static::generateUniqueSlug($category->name);
            }
        });

        static::updating(function ($category) {
            if ($category->isDirty('name') && ! $category->isDirty('slug')) {
                $category->slug = static::generateUniqueSlug($category->name);
            }
        });
    }

    public static function generateUniqueSlug($name)
    {
        $slug = Str::slug($name);
        $count = self::where('slug', 'like', "{$slug}%")->count();
        if ($count > 0) {
            $slug .= '-'.($count + 1);
        }

        return $slug;
    }
}
