<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class DebtPayment extends Model
{
    use HasFactory;

    protected $fillable = [
        'debt_id',
        'user_id',
        'amount',
        'payment_date',
        'payment_method',
        'notes',
        'reference',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'payment_date' => 'date',
    ];

    // Payment method constants
    const PAYMENT_METHOD_CASH = 'cash';
    const PAYMENT_METHOD_BANK_TRANSFER = 'bank_transfer';
    const PAYMENT_METHOD_MOBILE_MONEY = 'mobile_money';
    const PAYMENT_METHOD_OTHER = 'other';

    const PAYMENT_METHODS = [
        ['id' => 'cash', 'name' => 'Espèces', 'slug' => 'cash'],
        ['id' => 'bank_transfer', 'name' => 'Virement bancaire', 'slug' => 'bank_transfer'],
        ['id' => 'mobile_money', 'name' => 'Mobile Money', 'slug' => 'mobile_money'],
        ['id' => 'other', 'name' => 'Autre', 'slug' => 'other'],
    ];
    /**
     * Get the debt that this payment belongs to.
     */
    public function debt(): BelongsTo
    {
        return $this->belongsTo(Debt::class);
    }

    /**
     * Get the user who recorded this payment.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the formatted payment amount.
     */
    public function getFormattedAmountAttribute(): string
    {
        return number_format($this->amount, 0, ',', ' ') . ' ' . \App\Models\Setting::DEFAULT_CURRENCY;
    }

    /**
     * Get the payment method label.
     */
    public function getPaymentMethodLabelAttribute(): string
    {
        return match ($this->payment_method) {
            self::PAYMENT_METHOD_CASH => 'Espèces',
            self::PAYMENT_METHOD_BANK_TRANSFER => 'Virement bancaire',
            self::PAYMENT_METHOD_MOBILE_MONEY => 'Mobile Money',
            self::PAYMENT_METHOD_OTHER => 'Autre',
            default => 'Inconnu',
        };
    }

    /**
     * Get the payment method icon.
     */
    public function getPaymentMethodIconAttribute(): string
    {
        return match ($this->payment_method) {
            self::PAYMENT_METHOD_CASH => 'ph ph-money',
            self::PAYMENT_METHOD_BANK_TRANSFER => 'ph ph-bank',
            self::PAYMENT_METHOD_MOBILE_MONEY => 'ph ph-device-mobile',
            self::PAYMENT_METHOD_OTHER => 'ph ph-credit-card',
            default => 'ph ph-question',
        };
    }

    /**
     * Scope to get payments by method.
     */
    public function scopeByMethod($query, string $method)
    {
        return $query->where('payment_method', $method);
    }

    /**
     * Scope to get payments for a specific date range.
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('payment_date', [$startDate, $endDate]);
    }

    /**
     * Scope to get payments for a specific debt.
     */
    public function scopeForDebt($query, int $debtId)
    {
        return $query->where('debt_id', $debtId);
    }
}