<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class ActivityLog extends Model
{
    protected $fillable = [
        'user_id',
        'action',
        'description',
        'model_type',
        'model_id',
        'ip_address',
        'user_agent',
        'url',
        'method',
        'properties',
        'severity',
        'country',
        'city',
    ];

    protected function casts(): array
    {
        return [
            'properties' => 'array',
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
        ];
    }

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function model(): MorphTo
    {
        return $this->morphTo('model', 'model_type', 'model_id');
    }

    // Constants for actions
    const ACTION_LOGIN = 'login';
    const ACTION_LOGOUT = 'logout';
    const ACTION_CREATE = 'create';
    const ACTION_UPDATE = 'update';
    const ACTION_DELETE = 'delete';
    const ACTION_VIEW = 'view';
    const ACTION_ACTIVATE = 'activate';
    const ACTION_DEACTIVATE = 'deactivate';
    const ACTION_EXPORT = 'export';
    const ACTION_IMPORT = 'import';
    const ACTION_PRINT = 'print';
    const ACTION_PASSWORD_CHANGE = 'password_change';
    const ACTION_PROFILE_UPDATE = 'profile_update';

    // Constants for severity levels
    const SEVERITY_INFO = 'info';
    const SEVERITY_WARNING = 'warning';
    const SEVERITY_ERROR = 'error';
    const SEVERITY_CRITICAL = 'critical';

    // Scopes
    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeByAction($query, $action)
    {
        return $query->where('action', $action);
    }

    public function scopeBySeverity($query, $severity)
    {
        return $query->where('severity', $severity);
    }

    public function scopeRecent($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    // Helper methods
    public function getSeverityBadgeAttribute(): string
    {
        return match ($this->severity) {
            self::SEVERITY_CRITICAL => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">Critique</span>',
            self::SEVERITY_ERROR => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">Erreur</span>',
            self::SEVERITY_WARNING => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">Avertissement</span>',
            default => '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Info</span>',
        };
    }

    public function getActionIconAttribute(): string
    {
        return match ($this->action) {
            self::ACTION_LOGIN => 'ph ph-sign-in',
            self::ACTION_LOGOUT => 'ph ph-sign-out',
            self::ACTION_CREATE => 'ph ph-plus-circle',
            self::ACTION_UPDATE => 'ph ph-pencil',
            self::ACTION_DELETE => 'ph ph-trash',
            self::ACTION_VIEW => 'ph ph-eye',
            self::ACTION_ACTIVATE => 'ph ph-play-circle',
            self::ACTION_DEACTIVATE => 'ph ph-pause-circle',
            self::ACTION_EXPORT => 'ph ph-download',
            self::ACTION_IMPORT => 'ph ph-upload',
            self::ACTION_PRINT => 'ph ph-printer',
            self::ACTION_PASSWORD_CHANGE => 'ph ph-lock',
            self::ACTION_PROFILE_UPDATE => 'ph ph-user-circle',
            default => 'ph ph-activity',
        };
    }
}
