<?php

namespace App\Livewire\Auth;

use App\Models\User;
use Illuminate\Validation\Rules\Password;
use Livewire\Component;

class Register extends Component
{
    public string $firstname = '';

    public string $lastname = '';

    public string $email = '';

    public string $password = '';

    public string $merchant_name = '';

    public bool $canRegister = false;

    public bool $loading = false;

    public function updated()
    {
        $this->canRegister = !empty($this->merchant_name) && !empty($this->firstname) && !empty($this->lastname) && !empty($this->email) && !empty($this->password);
    }

    public function register(): void
    {
        $this->loading = true;

        $validationRules = [
            'firstname' => ['required', 'string', 'max:255'],
            'lastname' => ['required', 'string', 'max:255'],
            'merchant_name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255'],
            'password' => ['required', Password::defaults()],
        ];

        $validated = $this->validate($validationRules);

        // Online registration via API
        $apiResult = register($validated['firstname'], $validated['lastname'], $validated['merchant_name'], $validated['email'], $validated['password']);

        $user = $apiResult['user'] ?? null;
        $token = $apiResult['token'] ?? null;

        if (! $user || ! $token) {
            $this->addError('login', 'Une erreur est survenue lors de l\'enregistrement de l\'utilisateur. Veuillez réessayer plus tard.');
            return;
        }

        // Clean database and setup merchant
        if (! cleanDatabase()) {
            $this->addError('login', 'Une erreur est survenue lors de la nettoyage et de la configuration du marchand. Veuillez réessayer plus tard.');
            return;
        }

        // Create or update user in local database
        if (! createOrUpdateUserToLocalDatabase($user, $validated['password'])) {
            $this->addError('login', 'Une erreur est survenue lors de l\'enregistrement de l\'utilisateur. Veuillez réessayer plus tard.');
            return;
        }

        // Initialize user session
        if (! initializeUserSession($user, $token)) {
            $this->addError('login', 'Une erreur est survenue lors de l\'initialisation de la session utilisateur. Veuillez réessayer plus tard.');
            return;
        }

        // Initialize merchant session
        if (! initializeMerchantSession($validated['merchant_name'])) {
            $this->addError('login', 'Une erreur est survenue lors de l\'initialisation de la session marchand. Veuillez réessayer plus tard.');
            return;
        }

        // Redirect to home
        $this->redirectRoute('otp.verify', navigate: true);
        $this->loading = false;
        return;
    }


    public function render()
    {
        return view('livewire.auth.register');
    }
}
