<?php

namespace App\Http\Controllers;

use App\Models\StockEntry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class StockEntryController extends Controller
{
    public function create(Request $request)
    {
        // Return create view or handle via Livewire
        return redirect()->route('stock-entries.index');
    }

    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer une entrée de stock.');
        }

        $validated = $request->validate([
            'product_id' => ['required', 'exists:products,id'],
            'supplier_id' => ['nullable', 'exists:suppliers,id'],
            'purchase_order_id' => ['nullable', 'exists:purchase_orders,id'],
            'quantity' => ['required', 'integer', 'min:1'],
            'unit_price' => ['required', 'numeric', 'min:0'],
            'expiry_date' => ['nullable', 'date'],
            'notes' => ['nullable', 'string'],
        ]);

        $stockEntry = StockEntry::create([
            'reference' => StockEntry::generateReference(),
            'product_id' => $validated['product_id'],
            'supplier_id' => $validated['supplier_id'] ?? null,
            'purchase_order_id' => $validated['purchase_order_id'] ?? null,
            'quantity' => $validated['quantity'],
            'unit_price' => $validated['unit_price'],
            'total_price' => $validated['quantity'] * $validated['unit_price'],
            'expiry_date' => $validated['expiry_date'] ?? null,
            'notes' => $validated['notes'] ?? null,
            'user_id' => Auth::id(),
            'status' => StockEntry::STATUS_DRAFT,
        ]);

        return redirect()->route('stock-entries.show', $stockEntry)->with('success', 'Entrée de stock créée avec succès.');
    }

    public function update(Request $request, StockEntry $stockEntry)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas modifier une entrée de stock.');
        }

        if ($stockEntry->status !== StockEntry::STATUS_DRAFT) {
            return back()->with('error', 'Vous ne pouvez pas modifier une entrée déjà complétée.');
        }

        $validated = $request->validate([
            'quantity' => ['required', 'integer', 'min:1'],
            'unit_price' => ['required', 'numeric', 'min:0'],
            'expiry_date' => ['nullable', 'date'],
            'notes' => ['nullable', 'string'],
        ]);

        $stockEntry->update([
            'quantity' => $validated['quantity'],
            'unit_price' => $validated['unit_price'],
            'total_price' => $validated['quantity'] * $validated['unit_price'],
            'expiry_date' => $validated['expiry_date'] ?? null,
            'notes' => $validated['notes'] ?? null,
        ]);

        return back()->with('success', 'Entrée de stock modifiée avec succès.');
    }

    public function complete(StockEntry $stockEntry)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas compléter une entrée de stock.');
        }

        if ($stockEntry->status === StockEntry::STATUS_COMPLETED) {
            return back()->with('error', 'Cette entrée est déjà complétée.');
        }

        DB::transaction(function () use ($stockEntry) {
            $product = $stockEntry->product;
            $product->increment('current_stock', $stockEntry->quantity);

            $stockEntry->update([
                'status' => StockEntry::STATUS_COMPLETED,
                'completed_at' => now(),
            ]);
        });

        return back()->with('success', 'Entrée de stock complétée avec succès.');
    }

    public function cancel(StockEntry $stockEntry)
    {
        if ($stockEntry->status === StockEntry::STATUS_COMPLETED) {
            return back()->with('error', 'Vous ne pouvez pas annuler une entrée déjà complétée.');
        }

        $stockEntry->update([
            'status' => StockEntry::STATUS_CANCELLED,
        ]);

        return back()->with('success', 'Entrée de stock annulée avec succès.');
    }

    public function destroy(StockEntry $stockEntry)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer une entrée de stock.');
        }

        if ($stockEntry->status === StockEntry::STATUS_COMPLETED) {
            return back()->with('error', 'Vous ne pouvez pas supprimer une entrée déjà complétée.');
        }

        $stockEntry->delete();

        return redirect()->route('stock-entries.index')->with('success', 'Entrée de stock supprimée avec succès.');
    }
}
