<?php

namespace App\Http\Controllers;

use App\Models\Sale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SaleController extends Controller
{
    public function update(Request $request, Sale $sale)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas modifier une vente.');
        }

        $validated = $request->validate([
            'payment_method_id' => ['nullable', 'exists:payment_methods,id'],
            'sale_type_id' => ['nullable', 'exists:sale_types,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        $updateData = array_filter($validated, function ($value) {
            return $value !== null && $value !== '';
        });

        if (! empty($updateData)) {
            $sale->update($updateData);
        }

        return redirect()->route('sales.show', $sale)->with('success', 'Vente modifiée avec succès!');
    }

    public function destroy(Sale $sale)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer une vente.');
        }

        try {
            $sale->delete();

            return redirect()->route('sales.index')->with('success', 'Vente supprimée avec succès.');
        } catch (\Exception $e) {
            return back()->with('error', 'Erreur lors de la suppression de la vente.');
        }
    }

    public function print(Sale $sale)
    {
        $sale->load(['customer', 'paymentMethod', 'saleType', 'user', 'items.product']);

        return view('livewire.sales.print', compact('sale'));
    }

    public function cancel(Sale $sale)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas annuler une vente.');
        }

        if ($sale->status === Sale::STATUS_CANCELLED) {
            return back()->with('error', 'Cette vente est déjà annulée.');
        }

        try {
            $sale->update(['status' => Sale::STATUS_CANCELLED]);

            return redirect()->back()->with('success', 'Vente annulée avec succès.');
        } catch (\Exception $e) {
            return back()->with('error', 'Erreur lors de l\'annulation de la vente.');
        }
    }
}
