<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ProductController extends Controller
{
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer un produit.');
        }

        $sku = Product::generateUniqueSKU($request->name);


        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2'],
            'category_id' => ['required', 'exists:categories,id'],
        ]);
        $product = Product::create([
            'name' => $validated['name'],
            'sku' => $sku,
            'category_id' => $validated['category_id'],
            'slug' => Product::generateUniqueSlug(),
        ]);
        return redirect()->route('products.edit', $product)->with('success', 'Produit créé avec succès.');
    }

    public function activate(Product $product)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas activer un produit.');
        }

        // Guard: ensure all required commercial fields exist before activation
        $check = $product->hasAllRequiredFields();
        if (! ($check['success'] ?? false)) {
            return redirect()->route('products.edit', $product)
                ->with('error', $check['message'].' — complétez ces informations avant l\'activation.');
        }

        // Additional business rules
        if ((float) $product->purchase_price <= 0) {
            return redirect()->route('products.edit', $product)
                ->with('error', 'Le P.A. (prix d\'achat) doit être supérieur à 0.');
        }

        if ((float) $product->selling_price <= 0) {
            return redirect()->route('products.edit', $product)
                ->with('error', 'Le P.V. (prix de vente) doit être supérieur à 0.');
        }

        if (empty($product->purchase_unit_id) || empty($product->sale_unit_id)) {
            return redirect()->route('products.edit', $product)
                ->with('error', 'Définissez l\'unité d\'achat et l\'unité de vente.');
        }

        if ((int) $product->conversion_factor < 0) {
            return redirect()->route('products.edit', $product)
                ->with('error', 'Le facteur de conversion doit être supérieur à 0.');
        }

        $product->update(['status' => Product::STATUS_ACTIVE]);

        return redirect()->route('products.show', $product)->with('success', 'Produit activé avec succès.');
    }

    public function deactivate(Product $product)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas désactiver un produit.');
        }
        $product->update(['status' => Product::STATUS_INACTIVE]);

        return back()->with('success', 'Produit désactivé avec succès.');
    }

    public function destroy(Product $product)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer un produit.');
        }
        if ($product->sales()->count() > 0 || $product->stockEntries()->count() > 0) {
            return back()->with('error', 'Vous ne pouvez pas supprimer un produit avec des ventes ou des entrées de stock.');
        }
        $product->delete();

        return redirect()->route('products.index')->with('success', 'Produit supprimé avec succès.');
    }
}
