<?php

namespace App\Http\Controllers;

use App\Models\PaymentMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PaymentMethodController extends Controller
{
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer une méthode de paiement.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2'],
        ]);

        $paymentMethod = PaymentMethod::create([
            'name' => $validated['name'],
            'code' => PaymentMethod::generateUniqueCode($validated['name']),
            'status' => false,
        ]);

        return redirect()->route('payment-methods.edit', $paymentMethod)->with('success', 'Méthode de paiement créée avec succès.');
    }

    public function update(Request $request, PaymentMethod $paymentMethod)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas modifier une méthode de paiement.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2'],
        ]);

        $paymentMethod->update([
            'name' => $validated['name'],
        ]);

        return redirect()->route('payment-methods.edit', $paymentMethod)->with('success', 'Méthode de paiement modifiée avec succès.');
    }

    public function activate(PaymentMethod $paymentMethod)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas activer une méthode de paiement.');
        }

        $paymentMethod->update(['status' => true]);

        return back()->with('success', 'Méthode de paiement activée avec succès.');
    }

    public function deactivate(PaymentMethod $paymentMethod)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas désactiver une méthode de paiement.');
        }

        $paymentMethod->update(['status' => false]);

        return back()->with('success', 'Méthode de paiement désactivée avec succès.');
    }

    public function destroy(PaymentMethod $paymentMethod)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer une méthode de paiement.');
        }

        if ($paymentMethod->sales()->count() > 0) {
            return back()->with('error', 'Vous ne pouvez pas supprimer une méthode de paiement avec des ventes.');
        }

        $paymentMethod->delete();

        return redirect()->route('payment-methods.index')->with('success', 'Méthode de paiement supprimée avec succès.');
    }
}
