<?php

namespace App\Http\Controllers;

use App\Models\Expense;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ExpenseController extends Controller
{
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer une dépense.');
        }

        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'amount' => ['required', 'numeric', 'min:0'],
            'expense_category_id' => ['required', 'exists:expense_categories,id'],
        ]);

        $expense = Expense::create([
            'title' => $validated['title'],
            'amount' => $validated['amount'],
            'expense_category_id' => $validated['expense_category_id'],
            'user_id' => Auth::id(),
            'expense_date' => now()->format('Y-m-d'),
            'description' => null, // Set description to null for now
        ]);

        return redirect()->route('expenses.show', $expense)->with('success', 'Dépense créée avec succès!');
    }
    public function approve(Expense $expense)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas approuver une dépense.');
        }

        if (!$expense->canBeApproved()) {
            return back()->with('error', 'Cette dépense ne peut pas être approuvée.');
        }

        $expense->update(['status' => Expense::STATUS_APPROVED]);

        return redirect()->back()->with('success', 'Dépense approuvée avec succès.');
    }

    public function reject(Expense $expense)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas rejeter une dépense.');
        }

        if (!$expense->canBeRejected()) {
            return back()->with('error', 'Cette dépense ne peut pas être rejetée.');
        }

        $expense->update(['status' => Expense::STATUS_DRAFT]); // Reject = back to draft

        return redirect()->back()->with('success', 'Dépense rejetée avec succès.');
    }

    public function markPaid(Expense $expense)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas marquer une dépense comme payée.');
        }

        if (!$expense->canBeMarkedAsPaid()) {
            return back()->with('error', 'Cette dépense ne peut pas être marquée comme payée.');
        }

        // For boolean status, we'll keep it as approved when marked as paid
        // You might want to add a separate 'paid' field or modify the schema
        return redirect()->back()->with('success', 'Dépense marquée comme payée avec succès.');
    }

    public function destroy(Expense $expense)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer une dépense.');
        }

        if (!$expense->isDraft()) {
            return back()->with('error', 'Seules les dépenses en brouillon peuvent être supprimées.');
        }

        $expense->delete();

        return redirect()->route('expenses.index')->with('success', 'Dépense supprimée avec succès.');
    }
}
