<?php

namespace App\Http\Controllers;

use App\Models\ExpenseCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class ExpenseCategoryController extends Controller
{
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer une catégorie de dépense.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2', 'max:255', 'unique:expense_categories,name'],
        ]);

        $expenseCategory = ExpenseCategory::create([
            'slug' => Str::orderedUuid(),
            'name' => $validated['name'],
            'status' => ExpenseCategory::STATUS_INACTIVE,
        ]);

        return redirect()->route('expense-categories.edit', $expenseCategory)->with('success', 'Catégorie de dépense créée avec succès.');
    }

    public function activate(ExpenseCategory $category)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas activer une catégorie de dépense.');
        }

        $category->update(['status' => ExpenseCategory::STATUS_ACTIVE]);

        return redirect()->back()->with('success', 'Catégorie activée avec succès.');
    }

    public function deactivate(ExpenseCategory $category)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas désactiver une catégorie de dépense.');
        }

        $category->update(['status' => ExpenseCategory::STATUS_INACTIVE]);

        return redirect()->back()->with('success', 'Catégorie désactivée avec succès.');
    }

    public function destroy(ExpenseCategory $category)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer une catégorie de dépense.');
        }

        if ($category->expenses()->count() > 0) {
            return back()->with('error', 'Cette catégorie ne peut pas être supprimée car elle contient des dépenses.');
        }

        $category->delete();

        return redirect()->route('expense-categories.index')->with('success', 'Catégorie supprimée avec succès.');
    }
}
